<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

requireLogin();
$activePage = 'delivery';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Check permissions
$isSuperAdmin = false;
if (isset($_SESSION['user_id'])) {
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
}

$userName = $_SESSION['username'] ?? 'User';

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get deliveries (handle case where table doesn't exist)
$deliveries = [];
try {
    $stmt = $pdo->prepare("
        SELECT 
            d.*,
            c.name as customer_name,
            c.phone as customer_phone,
            c.address as customer_address,
            u.username as assigned_to_name,
            q.quote_number as quotation_number
        FROM deliveries d
        LEFT JOIN customers c ON d.customer_id = c.id
        LEFT JOIN users u ON d.assigned_to = u.id
        LEFT JOIN quotations q ON d.quotation_id = q.id
        ORDER BY d.created_at DESC
        LIMIT 50
    ");
    $stmt->execute();
    $deliveries = $stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist yet - will be created later
    $deliveries = [];
}

// Add search bar above deliveries table
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $search = strtolower(trim($_GET['search']));
    $deliveries = array_filter($deliveries, function($d) use ($search) {
        return strpos(strtolower($d['customer_name']), $search) !== false
            || strpos(strtolower($d['delivery_number']), $search) !== false
            || strpos(strtolower($d['status']), $search) !== false;
    });
}

// Get customers for dropdown
$stmt = $pdo->prepare("SELECT id, name, phone, address FROM customers ORDER BY name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Get users for assignment dropdown
$stmt = $pdo->prepare("SELECT id, username, full_name FROM users ORDER BY username");
$stmt->execute();
$users = $stmt->fetchAll();

// Get sales for delivery assignment
$stmt = $pdo->prepare("SELECT id, total_amount, created_at FROM sales WHERE status = 'completed' ORDER BY created_at DESC LIMIT 20");
$stmt->execute();
$sales = $stmt->fetchAll();

// Get active quotations for delivery scheduling
$quotations = [];
try {
    $stmt = $pdo->prepare("
        SELECT 
            q.id,
            q.quote_number,
            q.customer_id,
            q.total_amount,
            q.notes,
            q.created_at,
            c.name as customer_name,
            c.phone as customer_phone,
            c.address as customer_address
        FROM quotations q
        LEFT JOIN customers c ON q.customer_id = c.id
        WHERE q.status = 'active' AND q.valid_until >= CURDATE()
        ORDER BY q.created_at DESC
    ");
    $stmt->execute();
    $quotations = $stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist yet
    $quotations = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Delivery Management - <?php echo htmlspecialchars($settings['store_name'] ?? 'POS System'); ?></title>
    
    <!-- Favicon -->
    <?php echo getFaviconLink($pdo); ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    
    <!-- Custom CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    
    <style>
        :root {
            --primary-color: #2563eb;
            --primary-dark: #1d4ed8;
            --secondary-color: #64748b;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #06b6d4;
            --light-bg: #f8fafc;
            --border-color: #e2e8f0;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * {
            font-family: 'Inter', sans-serif;
        }

        body {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        .main-content {
            margin-left: 280px;
            padding: 2rem;
            transition: all 0.3s ease;
        }
        
        @media (max-width: 991px) {
            .main-content {
                margin-left: 0;
                padding: 1rem;
                padding-top: 80px;
            }
        }

        /* Header Section */
        .page-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 20px;
            padding: 1rem;
            margin-bottom: 2rem;
            color: white;
            box-shadow: var(--shadow-lg);
            position: relative;
            overflow: hidden;
        }

        .page-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            opacity: 0.3;
        }

        .page-header h1 {
            font-size: 1.75rem;
            font-weight: 700;
            margin: 0;
            position: relative;
            z-index: 1;
        }

        .page-header p {
            font-size: 0.9rem;
            opacity: 0.9;
            margin: 0.25rem 0 0 0;
            position: relative;
            z-index: 1;
        }

        .header-actions {
            position: relative;
            z-index: 1;
        }

        /* Cards */
        .card {
            border: none;
            border-radius: 16px;
            box-shadow: var(--shadow-md);
            background: white;
            transition: all 0.3s ease;
            overflow: hidden;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .card-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            padding: 0.75rem;
            border-radius: 16px 16px 0 0 !important;
        }

        .card-header h5 {
            font-weight: 600;
            color: var(--text-primary);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1rem;
        }

        .card-body {
            padding: 0.75rem;
        }

        /* Buttons */
        .btn {
            border-radius: 12px;
            font-weight: 500;
            padding: 0.75rem 1.5rem;
            transition: all 0.3s ease;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
        }

        .btn-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%);
            color: white;
        }

        .btn-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%);
            color: white;
        }

        .btn-outline-primary {
            border: 2px solid var(--primary-color);
            color: var(--primary-color);
            background: transparent;
        }

        .btn-outline-primary:hover {
            background: var(--primary-color);
            color: white;
            transform: translateY(-2px);
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 8px;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 0.375rem;
            box-shadow: var(--shadow-md);
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary-color);
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .stat-card .stat-icon {
            width: 25px;
            height: 25px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.25rem;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            font-size: 0.75rem;
        }

        .stat-card .stat-value {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.0625rem;
        }

        .stat-card .stat-label {
            color: var(--text-secondary);
            font-size: 0.625rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Table */
        .table {
            margin: 0;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border: none;
            padding: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            text-transform: uppercase;
            font-size: 0.75rem;
            letter-spacing: 0.5px;
        }

        .table td {
            padding: 0.5rem;
            border: none;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }

        .table tbody tr {
            transition: all 0.3s ease;
        }

        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            transform: scale(1.01);
        }

        .table tbody tr:last-child td {
            border-bottom: none;
        }

        /* Badges */
        .badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 500;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .bg-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%) !important;
        }

        .bg-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%) !important;
        }

        .bg-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%) !important;
            color: white !important;
        }

        .bg-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%) !important;
        }

        /* Action Buttons */
        .btn-group {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .btn-group .btn {
            border-radius: 8px;
            padding: 0.5rem;
            min-width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 3rem 2rem;
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: 4rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .empty-state h5 {
            font-size: 1.25rem;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .empty-state p {
            font-size: 1rem;
            margin: 0;
        }

        /* Form Controls */
        .form-control, .form-select {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem;
            transition: all 0.3s ease;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .form-label {
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        /* Modal Styling */
        .modal-content {
            border-radius: 20px;
            border: none;
            box-shadow: var(--shadow-xl);
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
        }

        .modal-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            border-radius: 20px 20px 0 0;
            border: none;
            padding: 1.5rem 2rem;
        }

        .modal-body {
            padding: 2rem;
        }

        .modal-footer {
            border: none;
            padding: 1.5rem 2rem;
            background: #f8fafc;
            border-radius: 0 0 20px 20px;
        }

        /* Search Bar */
        .search-container {
            position: relative;
            max-width: 400px;
        }

        .search-container .form-control {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem 0.75rem 3rem;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .search-container .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .search-container .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            z-index: 2;
        }

        /* Quotation Selection */
        .quotation-selection {
            border: 2px solid var(--primary-color);
            border-radius: 15px;
            background: linear-gradient(135deg, rgba(99, 102, 241, 0.05) 0%, rgba(99, 102, 241, 0.02) 100%);
            transition: all 0.3s ease;
        }
        
        .quotation-selected {
            border-color: var(--success-color);
            background: linear-gradient(135deg, rgba(16, 185, 129, 0.1) 0%, rgba(16, 185, 129, 0.05) 100%);
            box-shadow: 0 0 20px rgba(16, 185, 129, 0.2);
        }

        /* Animations */
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        @keyframes slideInRight {
            from {
                opacity: 0;
                transform: translateX(30px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }

        .fade-in-up {
            animation: fadeInUp 0.6s ease-out;
        }

        .slide-in-right {
            animation: slideInRight 0.6s ease-out;
        }

        /* Loading States */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .stats-card {
                margin-bottom: 1rem;
            }
            
            .table-responsive {
                border-radius: 15px;
                overflow: hidden;
            }
            
            .btn-group {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .btn-group .btn {
                border-radius: 8px !important;
            }
        }

        /* Modal Enhancements */
        .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            border-radius: 20px 20px 0 0;
            padding: 1.5rem;
        }

        .modal-title {
            font-weight: 600;
            color: var(--text-primary);
        }

        .modal-body {
            padding: 1.5rem;
        }

        .modal-footer {
            border-top: 1px solid var(--border-color);
            padding: 1.5rem;
            border-radius: 0 0 20px 20px;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .page-header h1 {
                font-size: 2rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .btn-group {
                flex-direction: column;
            }
            
            .btn-group .btn {
                width: 100%;
            }
        }

        /* Loading Animation */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Alert Enhancements */
        .alert {
            border: none;
            border-radius: 12px;
            padding: 1rem 1.5rem;
            margin-bottom: 1rem;
            box-shadow: var(--shadow-sm);
        }

        .alert-success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
        }

        .alert-danger {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
        }

        .alert-warning {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
        }

        .alert-info {
            background: linear-gradient(135deg, #cffafe 0%, #a5f3fc 100%);
            color: #155e75;
        }
    </style>
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>
    <?php include '../includes/topbar.php'; ?>
    
    <!-- Alert Container -->
    <div id="alertContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>
    
    <div class="main-content">
        <!-- Page Header -->
        <div class="page-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-truck me-3"></i>Deliveries</h1>
                    <p>Manage and track all your business deliveries</p>
                </div>
                <div class="header-actions">
            <?php if (hasPermission('add_delivery') || hasPermission('view_delivery')): ?>
                    <button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addDeliveryModal">
                        <i class="fas fa-plus"></i>
                        Create Delivery
            </button>
            <?php endif; ?>
        </div>
                    </div>
                </div>
        
        <!-- Stats Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-truck"></i>
            </div>
                <div class="stat-value"><?php echo count($deliveries); ?></div>
                <div class="stat-label">Total Deliveries</div>
                    </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);">
                    <i class="fas fa-clock"></i>
                </div>
                <div class="stat-value"><?php echo count(array_filter($deliveries, fn($d) => $d['status'] === 'pending')); ?></div>
                <div class="stat-label">Pending Deliveries</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%);">
                    <i class="fas fa-shipping-fast"></i>
                    </div>
                <div class="stat-value"><?php echo count(array_filter($deliveries, fn($d) => $d['status'] === 'in-transit')); ?></div>
                <div class="stat-label">In Transit</div>
                </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);">
                    <i class="fas fa-check-circle"></i>
            </div>
                <div class="stat-value"><?php echo count(array_filter($deliveries, fn($d) => $d['status'] === 'delivered')); ?></div>
                <div class="stat-label">Delivered</div>
            </div>
        </div>
        
        <!-- Deliveries Table -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5>
                        <i class="fas fa-list"></i>
                        All Deliveries
                </h5>
                    <div class="search-container">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" class="form-control" id="deliverySearchInput" placeholder="Search deliveries...">
            </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Delivery #</th>
                                <th>Customer</th>
                                <th>Address</th>
                                <th>Phone</th>
                                <th>Status</th>
                                <th>Assigned To</th>
                                <th>Scheduled Date</th>
                                <th>Quotation</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($deliveries)): ?>
                                <tr>
                                    <td colspan="9">
                                        <div class="empty-state">
                                            <i class="fas fa-truck"></i>
                                            <h5>No deliveries found</h5>
                                            <p>Create your first delivery to get started!</p>
                                            <button class="btn btn-primary mt-3" data-bs-toggle="modal" data-bs-target="#addDeliveryModal">
                                                <i class="fas fa-plus me-2"></i>Create Delivery
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($deliveries as $delivery): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($delivery['delivery_number']); ?></td>
                                        <td><?php echo htmlspecialchars($delivery['customer_name'] ?? 'Walk-in Customer'); ?></td>
                                        <td><?php echo htmlspecialchars($delivery['delivery_address']); ?></td>
                                        <td><?php echo htmlspecialchars($delivery['customer_phone'] ?? 'N/A'); ?></td>
                                        <td>
                                            <span class="badge <?php 
                                                echo match($delivery['status']) {
                                                    'pending' => 'bg-warning',
                                                    'in-transit' => 'bg-info',
                                                    'delivered' => 'bg-success',
                                                    'cancelled' => 'bg-danger',
                                                    default => 'bg-secondary'
                                                };
                                            ?>">
                                                <?php echo ucfirst(str_replace('-', ' ', $delivery['status'])); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($delivery['assigned_to_name'] ?? 'Unassigned'); ?></td>
                                        <td><?php echo date('M j, Y H:i', strtotime($delivery['scheduled_date'])); ?></td>
                                        <td>
                                            <?php if ($delivery['quotation_number']): ?>
                                                <span class="badge bg-info">
                                                    <i class="fas fa-file-invoice me-1"></i>
                                                    <?php echo htmlspecialchars($delivery['quotation_number']); ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <?php if (hasPermission('view_delivery')): ?>
                                                <button class="btn btn-outline-primary view-delivery" 
                                                        title="View Details"
                                                        data-id="<?php echo $delivery['id']; ?>"
                                                        data-delivery-number="<?php echo htmlspecialchars($delivery['delivery_number']); ?>"
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#viewDeliveryModal">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('mark_delivery_in_transit')): ?>
                                                <button class="btn btn-outline-info mark-in-transit" 
                                                        title="Mark as In Transit"
                                                        data-id="<?php echo $delivery['id']; ?>"
                                                        data-delivery-number="<?php echo htmlspecialchars($delivery['delivery_number']); ?>"
                                                        <?php echo $delivery['status'] !== 'pending' ? 'disabled' : ''; ?>>
                                                    <i class="fas fa-shipping-fast"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('mark_delivery_delivered')): ?>
                                                <button class="btn btn-outline-success mark-delivered" 
                                                        title="Mark as Delivered"
                                                        data-id="<?php echo $delivery['id']; ?>"
                                                        data-delivery-number="<?php echo htmlspecialchars($delivery['delivery_number']); ?>"
                                                        <?php echo $delivery['status'] === 'delivered' || $delivery['status'] === 'cancelled' ? 'disabled' : ''; ?>>
                                                    <i class="fas fa-check"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('cancel_delivery')): ?>
                                                <button class="btn btn-outline-danger cancel-delivery" 
                                                        title="Cancel Delivery"
                                                        data-id="<?php echo $delivery['id']; ?>"
                                                        data-delivery-number="<?php echo htmlspecialchars($delivery['delivery_number']); ?>"
                                                        <?php echo $delivery['status'] === 'delivered' || $delivery['status'] === 'cancelled' ? 'disabled' : ''; ?>>
                                                    <i class="fas fa-times"></i>
                                                </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Delivery Modal -->
    <div class="modal fade" id="addDeliveryModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-truck me-2"></i>
                        Schedule New Delivery
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="addDeliveryForm">
                    <div class="modal-body">
                        <!-- Quotation Selection -->
                        <div class="row mb-3">
                            <div class="col-12">
                                <div class="card quotation-selection" id="quotationSelectionCard">
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="quotation_id" class="form-label">
                                                        <i class="fas fa-file-invoice me-2"></i>Select from Quotation (Optional)
                                                    </label>
                                                    <select class="form-select" id="quotation_id" name="quotation_id">
                                                        <option value="">Create new delivery</option>
                                                        <?php if (empty($quotations)): ?>
                                                            <option value="" disabled>No active quotations available</option>
                                                        <?php else: ?>
                                                            <?php foreach ($quotations as $quotation): ?>
                                                                <option value="<?php echo $quotation['id']; ?>" 
                                                                        data-customer-id="<?php echo $quotation['customer_id'] ?? ''; ?>"
                                                                        data-customer-name="<?php echo htmlspecialchars($quotation['customer_name'] ?? 'Walk-in Customer'); ?>"
                                                                        data-customer-phone="<?php echo htmlspecialchars($quotation['customer_phone'] ?? ''); ?>"
                                                                        data-customer-address="<?php echo htmlspecialchars($quotation['customer_address'] ?? ''); ?>"
                                                                        data-total-amount="<?php echo $quotation['total_amount']; ?>"
                                                                        data-notes="<?php echo htmlspecialchars($quotation['notes'] ?? ''); ?>">
                                                                    <?php echo htmlspecialchars($quotation['quote_number']); ?> - 
                                                                    <?php echo htmlspecialchars($quotation['customer_name'] ?? 'Walk-in Customer'); ?> 
                                                                    (Ksh <?php echo number_format($quotation['total_amount'], 2); ?>)
                                                                </option>
                                                            <?php endforeach; ?>
                                                        <?php endif; ?>
                                                    </select>
                                                    <div class="form-text">
                                                        <i class="fas fa-info-circle me-1"></i>
                                                        Select a quotation to automatically populate customer details and delivery information.
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label class="form-label">Quotation Details</label>
                                                    <div id="quotationDetails" class="alert alert-info" style="display: none;">
                                                        <div id="quotationInfo"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="customer_id" class="form-label">Customer</label>
                                    <select class="form-select" id="customer_id" name="customer_id" required>
                                        <option value="">Select Customer</option>
                                        <?php foreach ($customers as $customer): ?>
                                            <option value="<?php echo $customer['id']; ?>" 
                                                    data-phone="<?php echo htmlspecialchars($customer['phone'] ?? ''); ?>"
                                                    data-address="<?php echo htmlspecialchars($customer['address'] ?? ''); ?>">
                                                <?php echo htmlspecialchars($customer['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="assigned_to" class="form-label">Assign To</label>
                                    <select class="form-select" id="assigned_to" name="assigned_to">
                                        <option value="">Select Delivery Person</option>
                                        <?php foreach ($users as $user): ?>
                                            <option value="<?php echo $user['id']; ?>">
                                                <?php echo htmlspecialchars($user['full_name'] ?? $user['username']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="delivery_phone" class="form-label">Phone Number</label>
                                    <input type="tel" class="form-control" id="delivery_phone" name="delivery_phone" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="scheduled_date" class="form-label">Scheduled Date & Time</label>
                                    <input type="datetime-local" class="form-control" id="scheduled_date" name="scheduled_date" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                        <div class="mb-3">
                            <label for="delivery_address" class="form-label">Delivery Address</label>
                            <textarea class="form-control" id="delivery_address" name="delivery_address" rows="3" required placeholder="Enter full delivery address..."></textarea>
                        </div>
                            </div>
                            <div class="col-md-6">
                        <div class="mb-3">
                            <label for="delivery_notes" class="form-label">Delivery Notes</label>
                            <textarea class="form-control" id="delivery_notes" name="delivery_notes" rows="3" placeholder="Special instructions or notes..."></textarea>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Quotation Reference (hidden field) -->
                        <input type="hidden" id="quotation_reference" name="quotation_reference" value="">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="delivery_fee" class="form-label">Delivery Fee</label>
                                    <input type="number" class="form-control" id="delivery_fee" name="delivery_fee" step="0.01" placeholder="0.00">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="payment_method" class="form-label">Payment Method</label>
                                    <select class="form-select" id="payment_method" name="payment_method">
                                        <option value="cash">Cash on Delivery</option>
                                        <option value="paid">Already Paid</option>
                                        <option value="credit">Credit</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Schedule Delivery</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Delivery Modal -->
    <div class="modal fade" id="viewDeliveryModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-eye me-2"></i>
                        View Delivery Details
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="deliveryDetails">
                        <div class="text-center py-4">
                            <i class="fas fa-spinner fa-spin fa-2x text-muted"></i>
                            <p class="text-muted mt-2">Loading delivery details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="printDelivery">
                        <i class="fas fa-print"></i> Print
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom JavaScript -->
    <script>
        // Modern Alert system with animations
        function showAlert(message, type = 'info') {
            const alertContainer = document.getElementById('alertContainer');
            const alertId = 'alert-' + Date.now();
            
            let alertClass = 'alert-info';
            let iconClass = 'fas fa-info-circle';
            let bgColor = 'linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%)';
            
            if (type === 'success') {
                alertClass = 'alert-success';
                iconClass = 'fas fa-check-circle';
                bgColor = 'linear-gradient(135deg, var(--success-color) 0%, #059669 100%)';
            } else if (type === 'error') {
                alertClass = 'alert-danger';
                iconClass = 'fas fa-exclamation-circle';
                bgColor = 'linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%)';
            } else if (type === 'warning') {
                alertClass = 'alert-warning';
                iconClass = 'fas fa-exclamation-triangle';
                bgColor = 'linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%)';
            }
            
            const alertHTML = `
                <div id="${alertId}" class="alert ${alertClass} alert-dismissible fade show" role="alert" 
                     style="background: ${bgColor}; color: white; border: none; border-radius: 15px; box-shadow: var(--shadow-lg); 
                            transform: translateX(100%); transition: transform 0.3s ease;">
                    <i class="${iconClass} me-2"></i>
                    <strong>${message}</strong>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            `;
            
            alertContainer.insertAdjacentHTML('beforeend', alertHTML);
            
            // Animate in
            setTimeout(() => {
                const alert = document.getElementById(alertId);
                if (alert) {
                    alert.style.transform = 'translateX(0)';
                }
            }, 100);
            
            // Auto-dismiss after 5 seconds
            setTimeout(() => {
                const alert = document.getElementById(alertId);
                if (alert) {
                    alert.style.transform = 'translateX(100%)';
                    setTimeout(() => {
                    const bsAlert = new bootstrap.Alert(alert);
                    bsAlert.close();
                    }, 300);
                }
            }, 5000);
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            // Enhanced AJAX Search functionality with modern UI
            const searchInput = document.getElementById('deliverySearchInput');
            if (searchInput) {
                let searchTimeout;
                let searchIcon = searchInput.parentElement.querySelector('.search-icon');
                
                searchInput.addEventListener('input', function() {
                    clearTimeout(searchTimeout);
                    const query = this.value.trim();
                    
                    // Show loading state
                    if (query.length > 0) {
                        searchIcon.className = 'fas fa-spinner fa-spin search-icon';
                        searchIcon.style.color = 'var(--primary-color)';
                    } else {
                        searchIcon.className = 'fas fa-search search-icon';
                        searchIcon.style.color = 'var(--primary-color)';
                    }
                    
                    // Show loading indicator
                    const tbody = document.querySelector('.table tbody');
                    if (query.length > 0) {
                        tbody.innerHTML = `
                            <tr>
                                <td colspan="9" class="text-center py-5">
                                    <div class="d-flex align-items-center justify-content-center">
                                        <div class="loading-spinner me-3"></div>
                                        <span class="text-muted">Searching deliveries...</span>
                                    </div>
                                </td>
                            </tr>
                        `;
                    }
                    
                    // Debounce search requests
                    searchTimeout = setTimeout(() => {
                        if (query.length === 0) {
                            location.reload(); // Reload to show all deliveries
                            return;
                        }
                        
                        fetch(`../api/search_deliveries.php?q=${encodeURIComponent(query)}`)
                            .then(res => res.json())
                            .then(data => {
                                searchIcon.className = 'fas fa-search search-icon';
                                searchIcon.style.color = 'var(--primary-color)';
                                
                                if (data.success) {
                                    updateDeliveryTable(data.deliveries);
                                } else {
                                    tbody.innerHTML = `
                                        <tr>
                                            <td colspan="9" class="text-center py-5">
                                                <i class="fas fa-exclamation-triangle fa-2x text-warning mb-3"></i>
                                                <p class="text-muted">Error: ${data.error || 'Search failed'}</p>
                                            </td>
                                        </tr>
                                    `;
                                }
                            })
                            .catch(error => {
                                searchIcon.className = 'fas fa-search search-icon';
                                searchIcon.style.color = 'var(--primary-color)';
                                tbody.innerHTML = `
                                    <tr>
                                        <td colspan="9" class="text-center py-5">
                                            <i class="fas fa-exclamation-triangle fa-2x text-danger mb-3"></i>
                                            <p class="text-muted">Error: ${error.message}</p>
                                        </td>
                                    </tr>
                                `;
                            });
                    }, 500);
                });
                
                // Add focus effects
                searchInput.addEventListener('focus', function() {
                    this.parentElement.style.transform = 'scale(1.02)';
                    this.parentElement.style.boxShadow = 'var(--shadow-lg)';
                });
                
                searchInput.addEventListener('blur', function() {
                    this.parentElement.style.transform = 'scale(1)';
                    this.parentElement.style.boxShadow = 'var(--shadow-md)';
                });
            }
            
            function updateDeliveryTable(deliveries) {
                const tbody = document.querySelector('.table tbody');
                
                if (deliveries.length === 0) {
                    tbody.innerHTML = '<tr><td colspan="8" class="text-center"><i class="fas fa-search fa-2x text-muted mb-2"></i><p class="text-muted">No deliveries found matching your search.</p></td></tr>';
                    return;
                }
                
                tbody.innerHTML = deliveries.map(delivery => `
                    <tr>
                        <td>${escapeHtml(delivery.delivery_number)}</td>
                        <td>${escapeHtml(delivery.customer_name || 'Walk-in Customer')}</td>
                        <td>${escapeHtml(delivery.quotation_reference || 'N/A')}</td>
                        <td>
                            <span class="badge ${getDeliveryStatusBadgeClass(delivery.status)}">
                                ${delivery.status.charAt(0).toUpperCase() + delivery.status.slice(1)}
                            </span>
                        </td>
                        <td>${formatDate(delivery.scheduled_date)}</td>
                        <td>${delivery.delivered_at ? formatDate(delivery.delivered_at) : 'Pending'}</td>
                        <td>${escapeHtml(delivery.assigned_to_name || 'Unassigned')}</td>
                        <td>
                            <div class="btn-group">
                                <button class="btn btn-sm btn-outline-primary view-delivery" title="View Delivery" data-id="${delivery.id}" data-delivery-number="${escapeHtml(delivery.delivery_number)}">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-info edit-delivery" title="Edit" data-id="${delivery.id}" data-delivery-number="${escapeHtml(delivery.delivery_number)}">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-success update-status" title="Update Status" data-id="${delivery.id}" data-status="${delivery.status}">
                                    <i class="fas fa-truck"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-warning" title="Print">
                                    <i class="fas fa-print"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger delete-delivery" title="Delete" data-id="${delivery.id}" data-delivery-number="${escapeHtml(delivery.delivery_number)}">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `).join('');
                
                // Re-attach event listeners for action buttons
                attachDeliveryActionListeners();
            }
            
            function getDeliveryStatusBadgeClass(status) {
                switch(status) {
                    case 'pending': return 'bg-warning';
                    case 'in-transit': return 'bg-info';
                    case 'delivered': return 'bg-success';
                    case 'cancelled': return 'bg-danger';
                    default: return 'bg-secondary';
                }
            }
            
            function formatDate(dateString) {
                return new Date(dateString).toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric'
                });
            }
            
            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }
            
            function attachDeliveryActionListeners() {
                // Re-attach view, edit, and delete event listeners for dynamically added rows
                // This will be handled by the existing document.addEventListener('click') handlers
            }
            
            // Set default scheduled date (tomorrow)
            const scheduledDateInput = document.getElementById('scheduled_date');
            const tomorrow = new Date();
            tomorrow.setDate(tomorrow.getDate() + 1);
            tomorrow.setHours(9, 0, 0, 0); // Set to 9:00 AM
            scheduledDateInput.value = tomorrow.toISOString().slice(0, 16);
            
            // Update delivery statistics
            updateDeliveryStatistics();
            
            // Check if there's a pre-selected quotation from quotation page
            const selectedQuotation = sessionStorage.getItem('selectedQuotation');
            if (selectedQuotation) {
                try {
                    const quotationData = JSON.parse(selectedQuotation);
                    
                    // Find and select the quotation in the dropdown
                    const quotationSelect = document.getElementById('quotation_id');
                    const quotationOption = Array.from(quotationSelect.options).find(option => 
                        option.value === quotationData.id
                    );
                    
                    if (quotationOption) {
                        quotationSelect.value = quotationData.id;
                        // Trigger change event to populate form
                        quotationSelect.dispatchEvent(new Event('change'));
                        
                        // Show success message
                        showAlert(`Quotation ${quotationData.quoteNumber} pre-selected for delivery scheduling`, 'success');
                        
                        // Open the modal automatically
                        const addDeliveryModal = new bootstrap.Modal(document.getElementById('addDeliveryModal'));
                        addDeliveryModal.show();
                    }
                    
                    // Clear the session storage
                    sessionStorage.removeItem('selectedQuotation');
                    
                } catch (error) {
                    console.error('Error parsing quotation data:', error);
                }
            }
            
            // Quotation selection change - populate form with quotation data
            document.getElementById('quotation_id').addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const quotationDetailsDiv = document.getElementById('quotationDetails');
                const quotationInfoDiv = document.getElementById('quotationInfo');
                const quotationReferenceInput = document.getElementById('quotation_reference');
                const quotationSelectionCard = document.getElementById('quotationSelectionCard');
                
                if (selectedOption.value) {
                    // Change card styling to show quotation is selected
                    quotationSelectionCard.classList.remove('quotation-selection');
                    quotationSelectionCard.classList.add('quotation-selected');
                    
                    // Populate customer selection
                    const customerSelect = document.getElementById('customer_id');
                    const customerId = selectedOption.dataset.customerId;
                    const customerName = selectedOption.dataset.customerName;
                    const customerPhone = selectedOption.dataset.customerPhone;
                    const customerAddress = selectedOption.dataset.customerAddress;
                    
                    // Always populate phone and address directly from quotation data first
                    const phoneInput = document.getElementById('delivery_phone');
                    const addressInput = document.getElementById('delivery_address');
                    
                    console.log('Quotation selected:', {
                        customerId: customerId,
                        customerPhone: customerPhone,
                        customerAddress: customerAddress,
                        customerName: customerName
                    });
                    
                    if (customerPhone) {
                        phoneInput.value = customerPhone;
                        console.log('Phone auto-filled:', customerPhone);
                    }
                    if (customerAddress) {
                        addressInput.value = customerAddress;
                        console.log('Address auto-filled:', customerAddress);
                    }
                    
                    if (customerId && customerId !== '') {
                        // If quotation has a specific customer, select that customer
                        customerSelect.dataset.programmaticChange = true;
                        customerSelect.value = customerId;
                        // Note: We don't trigger customer change event since we already populated phone/address
                    } else {
                        // If quotation is for walk-in customer, clear customer selection
                        customerSelect.value = '';
                    }
                    
                    // Add visual feedback for auto-filled fields
                    if (customerPhone || customerAddress) {
                        phoneInput.classList.add('border-success');
                        addressInput.classList.add('border-success');
                        setTimeout(() => {
                            phoneInput.classList.remove('border-success');
                            addressInput.classList.remove('border-success');
                        }, 2000);
                    }
                    
                    // Populate delivery notes with quotation notes
                    const deliveryNotesInput = document.getElementById('delivery_notes');
                    if (selectedOption.dataset.notes) {
                        deliveryNotesInput.value = `From Quotation: ${selectedOption.dataset.notes}`;
                    }
                    
                    // Set quotation reference
                    quotationReferenceInput.value = selectedOption.textContent.split(' - ')[0];
                    
                    // Show quotation details
                    const autoFilledInfo = [];
                    if (customerPhone) autoFilledInfo.push('Phone Number');
                    if (customerAddress) autoFilledInfo.push('Delivery Address');
                    if (selectedOption.dataset.notes) autoFilledInfo.push('Delivery Notes');
                    
                    const autoFilledText = autoFilledInfo.length > 0 ? 
                        `<small class="text-success"><i class="fas fa-check-circle"></i> Auto-filled: ${autoFilledInfo.join(', ')}</small>` : 
                        '<small class="text-muted"><i class="fas fa-info-circle"></i> No additional details to auto-fill</small>';
                    
                    quotationInfoDiv.innerHTML = `
                        <strong>Customer:</strong> ${selectedOption.dataset.customerName}<br>
                        <strong>Amount:</strong> Ksh ${parseFloat(selectedOption.dataset.totalAmount).toFixed(2)}<br>
                        <strong>Notes:</strong> ${selectedOption.dataset.notes || 'None'}<br>
                        ${autoFilledText}
                    `;
                    quotationDetailsDiv.style.display = 'block';
                    
                } else {
                    // Change card styling back to default
                    quotationSelectionCard.classList.remove('quotation-selected');
                    quotationSelectionCard.classList.add('quotation-selection');
                    
                    // Clear form and hide details
                    document.getElementById('customer_id').value = '';
                    document.getElementById('delivery_phone').value = '';
                    document.getElementById('delivery_address').value = '';
                    document.getElementById('delivery_notes').value = '';
                    quotationReferenceInput.value = '';
                    quotationDetailsDiv.style.display = 'none';
                    
                    console.log('Quotation deselected - form cleared');
                }
            });
            
            // Customer selection change - populate phone and address
            document.getElementById('customer_id').addEventListener('change', function() {
                const selectedOption = this.options[this.selectedIndex];
                const phoneInput = document.getElementById('delivery_phone');
                const addressInput = document.getElementById('delivery_address');
                
                if (selectedOption.dataset.phone) {
                    phoneInput.value = selectedOption.dataset.phone;
                }
                if (selectedOption.dataset.address) {
                    addressInput.value = selectedOption.dataset.address;
                }
                
                // Add visual feedback for auto-filled fields
                if (selectedOption.dataset.phone || selectedOption.dataset.address) {
                    phoneInput.classList.add('border-success');
                    addressInput.classList.add('border-success');
                    setTimeout(() => {
                        phoneInput.classList.remove('border-success');
                        addressInput.classList.remove('border-success');
                    }, 2000);
                }
                
                // If customer is manually changed, clear quotation selection
                // but only if the change wasn't triggered by quotation selection
                if (!this.dataset.programmaticChange) {
                    const quotationSelect = document.getElementById('quotation_id');
                    const quotationSelectionCard = document.getElementById('quotationSelectionCard');
                    const quotationDetailsDiv = document.getElementById('quotationDetails');
                    
                    if (quotationSelect.value) {
                        quotationSelect.value = '';
                        quotationSelectionCard.classList.remove('quotation-selected');
                        quotationSelectionCard.classList.add('quotation-selection');
                        quotationDetailsDiv.style.display = 'none';
                        document.getElementById('quotation_reference').value = '';
                    }
                }
                
                // Reset the flag
                this.dataset.programmaticChange = false;
            });
            
            // Form submission
            document.getElementById('addDeliveryForm').addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Validate form
                const form = e.target;
                const formData = new FormData(form);
                
                // Validate required fields
                const customerId = formData.get('customer_id');
                const deliveryPhone = formData.get('delivery_phone');
                const deliveryAddress = formData.get('delivery_address');
                const scheduledDate = formData.get('scheduled_date');
                
                if (!customerId || !deliveryPhone || !deliveryAddress || !scheduledDate) {
                    showAlert('Please fill in all required fields', 'error');
                    return;
                }
                
                // Show modern loading state
                const submitBtn = form.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<div class="loading-spinner me-2"></div>Scheduling Delivery...';
                submitBtn.style.background = 'linear-gradient(135deg, var(--primary-dark) 0%, var(--primary-color) 100%)';
                
                // Submit to API
                fetch('../api/create_delivery.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAlert(data.message, 'success');
                        
                        // Reset form
                        form.reset();
                        
                        // Reset quotation selection
                        const quotationSelectionCard = document.getElementById('quotationSelectionCard');
                        quotationSelectionCard.classList.remove('quotation-selected');
                        quotationSelectionCard.classList.add('quotation-selection');
                        document.getElementById('quotationDetails').style.display = 'none';
                        
                        // Reset scheduled date
                        const scheduledDateInput = document.getElementById('scheduled_date');
                        const tomorrow = new Date();
                        tomorrow.setDate(tomorrow.getDate() + 1);
                        tomorrow.setHours(9, 0, 0, 0);
                        scheduledDateInput.value = tomorrow.toISOString().slice(0, 16);
                        
                        // Close modal
                bootstrap.Modal.getInstance(document.getElementById('addDeliveryModal')).hide();
                        
                        // Refresh the page to show the new delivery
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                        
                    } else {
                        showAlert(data.message || 'Error scheduling delivery', 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Network error. Please try again.', 'error');
                })
                .finally(() => {
                    // Reset button state
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                    submitBtn.style.background = '';
                });
            });
            
            // Delivery action handlers
            
            // View delivery handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.view-delivery')) {
                    const btn = e.target.closest('.view-delivery');
                    const deliveryId = btn.dataset.id;
                    const deliveryNumber = btn.dataset.deliveryNumber;
                    
                    // Update modal title
                    document.querySelector('#viewDeliveryModal .modal-title').textContent = 'View Delivery - ' + deliveryNumber;
                    
                    // Load delivery details
                    fetch(`../api/get_delivery_details.php?id=${deliveryId}`)
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                displayDeliveryDetails(data.delivery, data.quotation_items);
                                window.currentDeliveryData = data.delivery; // Store for print
                                window.currentQuotationItems = data.quotation_items; // Store for print
                            } else {
                                document.getElementById('deliveryDetails').innerHTML = `
                                    <div class="alert alert-danger">
                                        <i class="fas fa-exclamation-circle"></i> ${data.message}
                                    </div>
                                `;
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            document.getElementById('deliveryDetails').innerHTML = `
                                <div class="alert alert-danger">
                                    <i class="fas fa-exclamation-circle"></i> Error loading delivery details
                                </div>
                            `;
                        });
                }
            });
            
            // Mark as in-transit handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.mark-in-transit')) {
                    const btn = e.target.closest('.mark-in-transit');
                    const deliveryId = btn.dataset.id;
                    const deliveryNumber = btn.dataset.deliveryNumber;
                    
                    if (confirm(`Mark delivery ${deliveryNumber} as in transit?`)) {
                        updateDeliveryStatus(deliveryId, 'in-transit', btn);
                    }
                }
            });
            
            // Mark as delivered handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.mark-delivered')) {
                    const btn = e.target.closest('.mark-delivered');
                    const deliveryId = btn.dataset.id;
                    const deliveryNumber = btn.dataset.deliveryNumber;
                    
                    if (confirm(`Mark delivery ${deliveryNumber} as delivered?`)) {
                        updateDeliveryStatus(deliveryId, 'delivered', btn);
                    }
                }
            });
            
            // Cancel delivery handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.cancel-delivery')) {
                    const btn = e.target.closest('.cancel-delivery');
                    const deliveryId = btn.dataset.id;
                    const deliveryNumber = btn.dataset.deliveryNumber;
                    
                    if (confirm(`Cancel delivery ${deliveryNumber}? This action cannot be undone.`)) {
                        updateDeliveryStatus(deliveryId, 'cancelled', btn);
                    }
                }
            });
            
            // Function to update delivery status
            function updateDeliveryStatus(deliveryId, status, btn) {
                // Show loading state
                btn.disabled = true;
                const originalHTML = btn.innerHTML;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                
                // Send update request
                const formData = new FormData();
                formData.append('delivery_id', deliveryId);
                formData.append('status', status);
                
                fetch('../api/update_delivery_status.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAlert(data.message, 'success');
                        // Refresh the page to show updated status
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                    } else {
                        showAlert(data.message, 'error');
                        // Reset button
                        btn.disabled = false;
                        btn.innerHTML = originalHTML;
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Error updating delivery status', 'error');
                    // Reset button
                    btn.disabled = false;
                    btn.innerHTML = originalHTML;
                });
            }
            
            // Function to display delivery details in modal
            function displayDeliveryDetails(delivery, quotationItems) {
                const customerInfo = delivery.customer_name ? `
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Customer:</strong> ${delivery.customer_name}<br>
                            <strong>Phone:</strong> ${delivery.customer_phone || 'N/A'}<br>
                            <strong>Email:</strong> ${delivery.customer_email || 'N/A'}
                        </div>
                        <div class="col-md-6">
                            <strong>Address:</strong><br>
                            ${delivery.customer_address || 'N/A'}
                        </div>
                    </div>
                ` : '<p><strong>Customer:</strong> Walk-in Customer</p>';
                
                const assignedInfo = delivery.assigned_to_name ? 
                    `${delivery.assigned_to_full_name || delivery.assigned_to_name}` : 'Unassigned';
                
                const deliveredInfo = delivery.delivered_by_name ? 
                    `${delivery.delivered_by_full_name || delivery.delivered_by_name} on ${new Date(delivery.delivered_at).toLocaleString()}` : 'Not delivered yet';
                
                let quotationInfo = '';
                if (delivery.quotation_number) {
                    quotationInfo = `
                        <div class="mb-3">
                            <h6>Related Quotation</h6>
                            <p><strong>Quote Number:</strong> ${delivery.quotation_number}</p>
                            <p><strong>Amount:</strong> Ksh ${parseFloat(delivery.quotation_amount).toFixed(2)}</p>
                            ${delivery.quotation_notes ? `<p><strong>Notes:</strong> ${delivery.quotation_notes}</p>` : ''}
                        </div>
                    `;
                }
                
                let itemsHtml = '';
                if (quotationItems.length > 0) {
                    itemsHtml = `
                        <h6>Items to Deliver</h6>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Item No</th>
                                        <th>Product</th>
                                        <th>Quantity</th>
                                        <th>Unit Price</th>
                                        <th>Total</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${quotationItems.map((item, idx) => `
                                        <tr>
                                            <td>${quotationItems.length}</td>
                                            <td>${item.product_name}</td>
                                            <td>${item.quantity}</td>
                                            <td>Ksh ${parseFloat(item.unit_price).toFixed(2)}</td>
                                            <td>Ksh ${parseFloat(item.total_price).toFixed(2)}</td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    `;
                }
                
                const html = `
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <h6>Delivery Information</h6>
                            <p><strong>Delivery Number:</strong> ${delivery.delivery_number}</p>
                            <p><strong>Status:</strong> <span class="badge ${delivery.status === 'pending' ? 'bg-warning' : (delivery.status === 'in-transit' ? 'bg-info' : (delivery.status === 'delivered' ? 'bg-success' : 'bg-danger'))}">${delivery.status.replace('-', ' ').charAt(0).toUpperCase() + delivery.status.replace('-', ' ').slice(1)}</span></p>
                            <p><strong>Scheduled Date:</strong> ${new Date(delivery.scheduled_date).toLocaleString()}</p>
                            <p><strong>Assigned To:</strong> ${assignedInfo}</p>
                            <p><strong>Delivered By:</strong> ${deliveredInfo}</p>
                        </div>
                        <div class="col-md-6">
                            <h6>Customer Information</h6>
                            ${customerInfo}
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <h6>Delivery Details</h6>
                            <p><strong>Delivery Phone:</strong> ${delivery.delivery_phone}</p>
                            <p><strong>Delivery Address:</strong><br>${delivery.delivery_address}</p>
                            <p><strong>Delivery Fee:</strong> Ksh ${parseFloat(delivery.delivery_fee).toFixed(2)}</p>
                            <p><strong>Payment Method:</strong> ${delivery.payment_method.charAt(0).toUpperCase() + delivery.payment_method.slice(1)}</p>
                        </div>
                        <div class="col-md-6">
                            <h6>Timestamps</h6>
                            <p><strong>Created:</strong> ${new Date(delivery.created_at).toLocaleString()}</p>
                            <p><strong>Last Updated:</strong> ${new Date(delivery.updated_at).toLocaleString()}</p>
                        </div>
                    </div>
                    
                    ${quotationInfo}
                    
                    ${delivery.delivery_notes ? `
                        <div class="mb-3">
                            <h6>Delivery Notes</h6>
                            <p class="text-muted">${delivery.delivery_notes}</p>
                        </div>
                    ` : ''}
                    
                    ${itemsHtml}
                `;
                
                document.getElementById('deliveryDetails').innerHTML = html;
            }
            
            // Print delivery handler
            document.getElementById('printDelivery').addEventListener('click', function() {
                const delivery = window.currentDeliveryData || {};
                const quotationItems = window.currentQuotationItems || [];
                const storeSettings = {
                    name: '<?php echo htmlspecialchars($settings['store_name'] ?? 'POS System'); ?>',
                    tagline: '<?php echo htmlspecialchars($settings['store_tagline'] ?? ''); ?>',
                    address: '<?php echo htmlspecialchars($settings['store_address'] ?? ''); ?>',
                    phone: '<?php echo htmlspecialchars($settings['store_phone'] ?? ''); ?>',
                    email: '<?php echo htmlspecialchars($settings['store_email'] ?? ''); ?>',
                    website: '<?php echo htmlspecialchars($settings['website'] ?? ''); ?>',
                    logo: '<?php 
                        $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
                        $stmt->execute();
                        $logo = $stmt->fetch();
                        if ($logo && $logo['setting_binary']) {
                            $mimeType = 'image/jpeg';
                            if (strpos($logo['setting_value'], '.png') !== false) {
                                $mimeType = 'image/png';
                            } elseif (strpos($logo['setting_value'], '.gif') !== false) {
                                $mimeType = 'image/gif';
                            }
                            echo 'data:' . $mimeType . ';base64,' . base64_encode($logo['setting_binary']);
                        }
                    ?>'
                };
                const today = new Date();
                const orderDate = delivery.scheduled_date ? new Date(delivery.scheduled_date) : today;
                const dueDate = delivery.due_date ? new Date(delivery.due_date) : today;
                const paddedItems = [...quotationItems];
                while (paddedItems.length < 10) { paddedItems.push({}); }
                const printHtml = `
<!DOCTYPE html>
<html>
<head>
    <title>Delivery Note</title>
    <style>
        body { font-family: 'Segoe UI', Arial, sans-serif; background: #f7fafd; margin: 0; color: #1a2330; }
        .header-main { display: flex; justify-content: space-between; align-items: flex-start; padding: 32px 48px 0 48px; background: #fff; }
        .store-block { max-width: 60%; }
        .store-logo { max-width: 90px; max-height: 50px; margin-bottom: 10px; }
        .store-name { font-size: 1.5rem; font-weight: 700; color: #1a2330; margin-bottom: 2px; }
        .store-details { font-size: 1rem; color: #444; line-height: 1.5; margin-bottom: 0; }
        .delivery-note-box { background: #fff; border: 2px solid #3a7afe; color: #3a7afe; font-size: 2rem; font-weight: 700; padding: 10px 36px; border-radius: 8px; box-shadow: 0 2px 8px rgba(34,48,74,0.04); white-space: nowrap; margin-top: 10px; }
        .main { max-width: 900px; margin: 0 auto; background: #f7fafd; border-radius: 16px; box-shadow: 0 2px 12px rgba(34,48,74,0.07); padding: 0 32px 32px 32px; }
        .row-top { display: flex; justify-content: space-between; gap: 32px; margin-top: 32px; }
        .delivery-to-card { background: #fff; border-radius: 10px; box-shadow: 0 1px 4px rgba(34,48,74,0.04); padding: 20px 24px; flex: 1; min-width: 0; }
        .delivery-title { font-weight: 700; color: #1a2330; font-size: 1.1rem; margin-bottom: 10px; }
        .delivery-label { font-weight: 600; color: #3a7afe; font-size: 1rem; }
        .info-table-2col { width: 100%; border-collapse: collapse; background: #fff; border-radius: 10px; box-shadow: 0 1px 4px rgba(34,48,74,0.04); overflow: hidden; }
        .info-table-2col td { border: 1px solid #b2c0db; padding: 4px 14px; font-size: 1rem; text-align: left; font-weight: 500; }
        .info-table-2col .label { background: #eaf4ff; color: #1a2330; font-weight: 700; width: 40%; }
        .info-table-2col .value { background: #fff; color: #1a2330; width: 60%; }
        .items-table { width: 100%; border-collapse: collapse; margin-top: 32px; margin-bottom: 32px; background: #fff; border-radius: 10px; box-shadow: 0 1px 4px rgba(34,48,74,0.04); overflow: hidden; }
        .items-table th, .items-table td { border: 1px solid #b2c0db; padding: 10px 8px; font-size: 1rem; text-align: center; }
        .items-table th { background: #eaf4ff; color: #1a2330; font-weight: 700; }
        .items-table .item-no { width: 40px; }
        .items-table .qty-col { width: 50px; }
        .items-table td { background: #fff; height: 32px; }
        .items-table td:last-child { width: 60px; }
        .signature-row { display: flex; justify-content: space-between; margin-top: 40px; font-size: 1rem; }
        .signature-label { font-weight: 500; color: #1a2330; }
        .signature-line { display: inline-block; border-bottom: 1px dotted #b2c0db; width: 220px; height: 1.2em; margin-left: 8px; }
    </style>
</head>
<body>
    <div class="header-main">
        <div class="store-block">
            ${storeSettings.logo ? `<img src="${storeSettings.logo}" alt="Store Logo" class="store-logo">` : ''}
            <div class="store-name">${storeSettings.name}</div>
            <div class="store-details">
                Address: ${storeSettings.address}<br>
                Email: ${storeSettings.email}<br>
                Phone: ${storeSettings.phone}<br>
                ${storeSettings.website ? `Website: ${storeSettings.website}` : ''}
            </div>
        </div>
        <div class="delivery-note-box">Delivery Note</div>
    </div>
    <div class="main">
        <div class="row-top">
            <div class="delivery-to-card">
                <div class="delivery-title">Delivery to</div>
                <div><span class="delivery-label">To :</span> ${delivery.customer_name || ''}</div>
                <div><span class="delivery-label">Address:</span> ${delivery.delivery_address || ''}${delivery.city ? ', ' + delivery.city : ''}</div>
            </div>
            <div style="flex:1; display:flex; align-items:flex-start; justify-content:flex-end;">
                <table class="info-table-2col" style="min-width: 220px; max-width: 320px; margin-top:0;">
                    <tr><td class="label">Quote No</td><td class="value">${delivery.quotation_reference || delivery.quotation_number || ''}</td></tr>
                    <tr><td class="label">Order No</td><td class="value">${delivery.order_number || ''}</td></tr>
                    <tr><td class="label">Order Date</td><td class="value">${orderDate ? orderDate.toLocaleDateString() : ''}</td></tr>
                    <tr><td class="label">Due Date</td><td class="value">${dueDate ? dueDate.toLocaleDateString() : ''}</td></tr>
                </table>
            </div>
        </div>
        <table class="items-table">
            <thead>
                <tr>
                    <th class="item-no">Item No</th>
                    <th>Description</th>
                    <th class="qty-col">QTY</th>
                    <th>Complete</th>
                </tr>
            </thead>
            <tbody>
                ${paddedItems.map((item, idx) => `
                    <tr>
                        <td class="item-no">${item.product_name || item.description ? (idx + 1) : ''}</td>
                        <td>${item.product_name || item.description || ''}</td>
                        <td>${item.quantity || ''}</td>
                        <td>${item.product_name || item.description ? `<input type='checkbox' style='pointer-events:none;transform:scale(1.2);' ${delivery.status === 'delivered' ? 'checked' : ''} />` : ''}</td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
        <div class="signature-row">
            <div><span class="signature-label">Date:</span> <span class="signature-line"></span></div>
            <div><span class="signature-label">Signature:</span> <span class="signature-line"></span></div>
        </div>
    </div>
</body>
</html>
                `;
                const printWindow = window.open('', '_blank', 'width=900,height=1200');
                printWindow.document.write(printHtml);
                printWindow.document.close();
                printWindow.onload = function() {
                    printWindow.print();
                    setTimeout(() => { printWindow.close(); }, 1000);
                };
            });
            
            // Function to update delivery statistics
            function updateDeliveryStatistics() {
                const deliveries = <?php echo json_encode($deliveries); ?>;
                const stats = {
                    pending: 0,
                    'in-transit': 0,
                    delivered: 0,
                    cancelled: 0
                };
                
                deliveries.forEach(delivery => {
                    if (stats.hasOwnProperty(delivery.status)) {
                        stats[delivery.status]++;
                    }
                });
                
                document.getElementById('pendingCount').textContent = stats.pending;
                document.getElementById('inTransitCount').textContent = stats['in-transit'];
                document.getElementById('deliveredCount').textContent = stats.delivered;
                document.getElementById('cancelledCount').textContent = stats.cancelled;
            }
        });
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 